<?php

if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly.
}

class Ultimate_AI_Subscription
{
  private $user_id;
  private $subscriber_id;
  private $subscription_id;
  public $UAI_PAYMENTS;
  public $UAI_ACTIVITY;
  public $UAI_STRIPE;
  public $UAI_PAYPAL;
  public $UAI_WOO;
  public $UAI_USER;

  public function __construct()
  {
    $this->UAI_USER = new Ultimate_AI_User_Manager();
    $this->UAI_PAYMENTS = new Ultimate_AI_Payments_Module();
    $this->UAI_STRIPE = new Ultimate_AI_Stripe();
    $this->UAI_WOO = new Ultimate_AI_WooCommerce();
    $this->UAI_ACTIVITY = new Ultimate_AI_Activity();
    $this->UAI_PAYPAL = new Ultimate_AI_Paypal();
  }

  public function get_subscriptions()
  {
    return get_option('ultimate_ai_subscription_packs', []);
  }

  public function updateSubcriptionStatus($status, $data)
  {
    switch ($status) {
      case 'CANCEL':
        // ORDER STATUS UPDATE
        update_post_meta($data['order_post_id'], 'ultimate_ai_order_meta', $data);
        update_post_meta($data['order_post_id'], 'subscribed', $data['subscribed']);
        update_post_meta($data['order_post_id'], 'starts', $data['starts']);
        update_post_meta($data['order_post_id'], 'expiry', $data['expiry']);
        update_post_meta($data['order_post_id'], 'payment_status', $data['payment_status']);

        $user_id = $this->UAI_USER->getUserBySubscriptionID($data['subscription_id']);

        $user = get_user_by('ID', $user_id);

        $this->UAI_USER->update_user_token($user, 0, 'word', 'INCREMENT', true);
        $this->UAI_USER->update_user_token($user, 0, 'image', 'INCREMENT', true);

        $added = $this->UAI_USER->update_user_meta($user, [
          'limits' => [
            'words' => 0,
            'images' => 0,
          ],
          'tools' => [
            'tools' => [],
            'advanced_tools' => [],
          ],
          'subscription' => $data,
          'user_active' => false,
        ]);

        // CANCEL FROM PAYMENT GATEWAY
        if (isset($data['subscription_id'])) {
          $this->UAI_PAYMENTS->updateGatewaySubscriptionStatus($data['payment_method'], 'CANCEL', $data['subscription_id']);
        }

        // SEND MAIL AFTER CANCELLING SUBSCRIPTION
        $mail_params = [
          'to' => $data['email'],

          // MAIL SUBJECT
          'subject' => esc_html__('Your subscription plan is cancelled!', ULTIMATE_AI_SLUG),

          // MAIL MESSAGE
          'message' => sprintf(esc_html__('Hello %1$s, Your %2$s subscription to %3$s is cancelled', ULTIMATE_AI_SLUG), $data['name'], $data['plan_interval'], $data['plan_name']),
        ];

        $sent = $this->UAI_USER->send_mail($mail_params);

        if ($added) {
          return true;
        }

        if (!$added) {
          return false;
        }
        break;
    }
  }

  public function processSubscriptonWebhook($gateway, $eventObj)
  {
    switch ($gateway) {
      case 'stripe':
        $type = $eventObj->type;

        switch ($type) {
          case 'customer.subscription.deleted':
          case 'invoice.payment_failed':
          case 'customer.subscription.updated':
            $subscription_id = $eventObj->data->object->id;
            $UAI_ORDERS = new Ultimate_AI_Orders();

            $order_post_id = $UAI_ORDERS->getOrderBySubscriptionID($subscription_id);

            $meta = get_post_meta($order_post_id, 'ultimate_ai_order_meta', true);

            $subscribed = $eventObj->data->object->status == 'active' ? true : false;
            $meta['order_post_id'] = $order_post_id;
            $meta['subscribed'] = $subscribed;
            $meta['starts'] = $eventObj->data->object->current_period_start;
            $meta['expiry'] = $eventObj->data->object->current_period_end;
            $meta['payment_status'] = $eventObj->data->object->status;

            $this->updateSubcriptionStatus('CANCEL', $meta);

            break;
        }
        break;

      case 'paypal':
        $type = $eventObj->event_type;

        switch ($type) {
          case 'BILLING.SUBSCRIPTION.EXPIRED':
          case 'BILLING.SUBSCRIPTION.CANCELLED':
          case 'BILLING.SUBSCRIPTION.SUSPENDED':
          case 'BILLING.SUBSCRIPTION.UPDATED':
            $subscription_id = $eventObj->resource->id;
            $UAI_ORDERS = new Ultimate_AI_Orders();

            $order_post_id = $UAI_ORDERS->getOrderBySubscriptionID($subscription_id);

            $meta = get_post_meta($order_post_id, 'ultimate_ai_order_meta', true);

            $subscribed = strtolower($eventObj->resource->status) == 'active' ? true : false;

            $meta['order_post_id'] = $order_post_id;
            $meta['subscribed'] = $subscribed;
            $meta['expiry'] = time();
            $meta['payment_status'] = strtolower($eventObj->resource->status);

            $this->updateSubcriptionStatus('CANCEL', $meta);
            break;
        }
        break;
    }

    return $meta;
  }

  public function create_subscription_pack($data)
  {
    if (empty($data['plan_name']) || empty($data['pricing']) || empty($data['tools'])) {
      wp_send_json([
        'message' => esc_html__('Plan name, pricing and tools are required', ULTIMATE_AI_SLUG),
        'loading' => false,
        'success' => false,
      ], 200);
    }
    $data['created_on_unix'] = time();
    $data['created_on'] = date('M d, Y', time());
    $data['status'] = 'active';
    $packs = $this->get_subscriptions();

    if (isset($data['id'])) {
      if (is_numeric($this->getSubscriptionPackIdx($data['id']))) {
        $packs[$this->getSubscriptionPackIdx($data['id'])] = $data;
      } else {
        $data['id'] = 'uai_subs_' . rand();
        $packs[] = $data;
      }
    } else {
      $data['id'] = 'uai_subs_' . rand();
      $packs[] = $data;
    }

    $updated = update_option('ultimate_ai_subscription_packs', $packs);

    if ($updated) {
      $res = [
        'message' => esc_html__('Successfull updated', ULTIMATE_AI_SLUG),
        'success' => true,
        'loading' => false,
        'items' => $this->get_subscriptions(),
      ];
    } else {
      $res = [
        'message' => esc_html__('Update failed', ULTIMATE_AI_SLUG),
        'success' => false,
        'loading' => false,
        'items' => [],
      ];
    }

    return $res;
  }

  public function delete_subscription_pack($index)
  {
    $packs = $this->get_subscriptions();

    unset($packs[$index]);
    $new_packs = array_values($packs);
    $updated = update_option('ultimate_ai_subscription_packs', $new_packs);
    if ($updated) {
      $res = [
        'message' => esc_html__('Successfully deleted', ULTIMATE_AI_SLUG),
        'success' => true,
        'loading' => false,
        'items' => $this->get_subscriptions(),
      ];
    } else {
      $res = [
        'message' => esc_html__('Deleting pricing failed.', ULTIMATE_AI_SLUG),
        'success' => false,
        'loading' => false,
        'items' => [],
      ];
    }

    return $res;
  }

  public function update_subscription_pack_status($index)
  {
    $packs = $this->get_subscriptions();

    $current_status = $packs[$index]['status'];
    if ($current_status == 'active') {
      $packs[$index]['status'] = 'inactive';
    } else {
      $packs[$index]['status'] = 'active';
    }
    $updated = update_option('ultimate_ai_subscription_packs', $packs);
    if ($updated) {
      $res = [
        'message' => esc_html__('Successfully status changed', ULTIMATE_AI_SLUG),
        'success' => true,
        'loading' => false,
        'items' => $this->get_subscriptions(),
      ];
    } else {
      $res = [
        'message' => esc_html__('Status change failed.', ULTIMATE_AI_SLUG),
        'success' => false,
        'loading' => false,
        'items' => [],
      ];
    }

    return $res;
  }

  /**
   * @since v1.0.0
   * Create subscription when order is successful
   * Accepts order $data from success page
   *
   * @return [
   *      'status' => 'CREATED',
   *      'data' => $data
   * ];
   */
  public function subscribe($data)
  {
    $exists = $this->orderAlreadyCreated($data);
    if ($exists) {
      $meta = get_post_meta($exists, 'ultimate_ai_order_meta', true);

      $meta['order_id'] = $exists;

      return [
        'status' => 'EXISTS',
        'data' => $meta,
      ];
    }

    $user = get_user_by('ID', $data['user_id']);

    $data['package_type'] = $this->getPackageType($data['pricing_id']);
    $data['package_name'] = $this->getPackageNameAndID($data['pricing_id'])['plan_name'];
    $data['payment_type'] = $this->UAI_USER->is_user_subscriber($user) ? 'Purchase' : 'Renew';

    // Get payment method
    $payment_method = $data['payment_method'];

    if ($payment_method == 'Self') {
      $product_id = '';
    } elseif ($payment_method == 'woocommerce') {
      $product_id = '';
      $data['plan_interval'] = (intval($data['expiry']) - intval($data['starts'])) > 360 * 24 * 60 * 60 ? 'Yearly' : 'Monthly';
    } else {
      if ($data['package_type'] == 'subscription') {
        $data['plan_interval'] = (intval($data['expiry']) - intval($data['starts'])) > 360 * 24 * 60 * 60 ? 'Yearly' : 'Monthly';

        $product_id = $data['plan_' . $payment_method . '_id'];
      } else {
        $data['plan_interval'] = '';
        $product_id = '';
      }
    }

    // Set Plan name and plan ID
    $data['plan_name'] = $this->getPackageNameAndID($data['pricing_id'])['plan_name'];
    $data['plan_id'] = $this->getPackageNameAndID($data['pricing_id'])['plan_id'];

    // Currency symbol
    $data['currency_symbol'] = $this->getCurrencySymbol($data['currency']);

    $UAI_ORDERS = new Ultimate_AI_Orders();

    // Creating order for this subscription
    $data['order_id'] = $UAI_ORDERS->createOrder($data);
    $data['activity_id'] = $this->UAI_ACTIVITY->createActivity([
      'subject_id' => $data['user_id'],
      'object_id' => 0,
      'action_type' => 'PURCHASE',
      'object_type' => 'SUBSCRIPTION',
      'object_name' => $this->getPackageNameAndID($data['pricing_id'])['plan_name'],
    ]);
    // Get package limits
    $limits = $this->getpackageLimits($data['pricing_id'], $data['plan_interval']);
    $data['limits'] = $limits;

    // Get package tools
    $tools = $this->getPackageTools($data['pricing_id']);

    $this->UAI_USER->update_user_token($user, $limits['words'], 'word', 'INCREMENT', true);
    $this->UAI_USER->update_user_token($user, $limits['images'], 'image', 'INCREMENT', true);

    $added = $this->UAI_USER->update_user_meta($user, [
      'limits' => $limits,
      'tools' => $tools,
      'subscription' => $data,
      'user_active' => $data['subscribed'],
    ]);

    // SEND MAIL AFTER CREATING SUBSCRIPTION
    $mail_params = [
      'to' => $data['email'],

      // MAIL SUBJECT
      'subject' => esc_html__('Congratulations!!! your subscription is confirmed!', ULTIMATE_AI_SLUG),

      // MAIL MESSAGE
      'message' => sprintf(esc_html__('Congratulations %1$s, Your %2$s subscription to %3$s is successful', ULTIMATE_AI_SLUG), $data['name'], $data['plan_interval'], $data['plan_name']),
    ];

    $this->UAI_USER->send_mail($mail_params);

    if (!is_wp_error($added)) {
      return [
        'status' => 'CREATED',
        'data' => $data,
      ];
    } else {
      return [
        'status' => 'FAIL',
        'data' => null,
      ];
    }
  }

  public function userPaymentType()
  {
  }

  public function orderAlreadyCreated($data)
  {
    if (!isset($data['order_id'])) {
      return false;
    }
    $order_id = $data['order_id'];

    $args = [
      'post_status' => 'private',
      'post_type' => 'ultimate_ai_orders',
      'fields' => 'ids',
      'meta_query' => [
        [
          'key' => 'order_id',
          'value' => $order_id,
        ],
      ],
    ];

    $posts = get_posts($args);

    if (count($posts) > 0) {
      return $posts[0];
    } else {
      return false;
    }
  }

  /**
   * @since v1.0.0
   * Get remaining words count if an user
   *
   * @accepts INT $user_id
   *
   * @return int $count
   */
  public function getRemainingWords($user_id)
  {
    $limits = get_user_meta($user_id, 'ultimate_ai_subscription_user_limits');

    return $limits['words'];
  }

  public function getRemainingImages($user_id)
  {
    $limits = get_user_meta($user_id, 'ultimate_ai_subscription_user_limits');

    return $limits['images'];
  }

  public function updateRemainingWords($user_id, $words_used)
  {
    $limits = get_user_meta($user_id, 'ultimate_ai_subscription_user_limits');
    $remaining = (int) $limits['words'] - (int) $words_used;
    $limits['words'] = $remaining;
    update_user_meta($user_id, 'ultimate_ai_subscription_user_limits', $limits);
  }

  public function updateRemainingImages($user_id, $images_used)
  {
    $limits = get_user_meta($user_id, 'ultimate_ai_subscription_user_limits');
    $remaining = (int) $limits['images'] - (int) $images_used;
    $limits['images'] = $remaining;
    update_user_meta($user_id, 'ultimate_ai_subscription_user_limits', $limits);
  }

  public function getPackageTools($pricing_id)
  {
    $packs = $this->get_subscriptions();
    $tools = [];
    $advanced_tools = [];
    $integration = [];
    foreach ($packs as $k => $pack) {
      foreach ($pack['pricing'] as $key => $price) {
        if ($price['id'] == $pricing_id) {
          $tools = $packs[$k]['tools'];
          $advanced_tools = $packs[$k]['advancedtools'];
          $integration = isset($packs[$k]['integration']) ? $packs[$k]['integration'] : [];
        }
      }
    }

    return [
      'tools' => $tools,
      'advanced_tools' => $advanced_tools,
      'integration' => $integration,
    ];
  }

  public function getPackageIntegrationAccess($pricing_id, $needle)
  {
    $packs = $this->get_subscriptions();
    $integration = [];
    foreach ($packs as $k => $pack) {
      foreach ($pack['pricing'] as $key => $price) {
        if ($price['id'] == $pricing_id) {
          if (isset($packs[$k]['integration'])) {
            $integration = $packs[$k]['integration'];
          }
        }
      }
    }
    if (is_array($integration) && !empty($integration)) {
      if (in_array($needle, $integration)) {
        return true;
      }
    } else {
      return false;
    }
  }

  /**
   * @since v1.0.0
   * Get package/plan Name or ID
   *
   * @accepts String $pricing_id
   *
   * @return array $plan_name and $plan_id
   */
  public function getPackageNameAndID($pricing_id)
  {
    $packs = $this->get_subscriptions();

    foreach ($packs as $k => $pack) {
      foreach ($pack['pricing'] as $key => $price) {
        if ($price['id'] == $pricing_id) {
          $plan_name = $pack['plan_name'];
          $plan_id = $pack['id'];
        }
      }
    }

    return [
      'plan_name' => $plan_name,
      'plan_id' => $plan_id,
    ];
  }

  /**
   * @since v1.0.0
   * Get package/plan Name or ID
   *
   * @accepts String $pricing_id
   *
   * @return array $plan_name and $plan_id
   */
  public function getSubscriptionPackIdx($id)
  {
    $packs = $this->get_subscriptions();
    foreach ($packs as $k => $pack) {
      if ($pack['id'] == $id) {
        return $k;
      }
    }

    return 'not_exists';
  }

  /**
   * @since v1.0.0
   * Get package/plan Name or ID
   *
   * @accepts String $pricing_id
   *
   * @return array $plan_name and $plan_id
   */
  public function getPricingID($plan_id, $index = 0)
  {
    $packs = $this->get_subscriptions();
    foreach ($packs as $k => $pack) {
      if ($pack['id'] == $plan_id) {
        return $packs[$k]['pricing'][$index]['id'];
      }
    }

    return false;
  }

  /**
   * @since v1.0.0
   * Get package/plan Name or ID
   *
   * @accepts String $pricing_id
   *
   * @return array $plan_name and $plan_id
   */
  public function getPackageType($pricing_id)
  {
    $packs = $this->get_subscriptions();

    $type = '';
    foreach ($packs as $k => $pack) {
      foreach ($pack['pricing'] as $key => $price) {
        if ($price['id'] == $pricing_id) {
          $type = $packs[$k]['type'];
        }
      }
    }

    return $type;
  }

  /**
   * @since v1.0.0
   * Get limits of the plan
   *
   * @accepts String $pricing_id
   *
   * @return array $limits
   */
  public function getpackageLimits($pricing_id, $interval = '')
  {
    $packs = $this->get_subscriptions();
    $limits = [];
    foreach ($packs as $k => $pack) {
      foreach ($pack['pricing'] as $key => $price) {
        if ($price['id'] == $pricing_id) {
          $type = $packs[$k]['type'];
          if ($type == 'subscription') {
            foreach ($price['schemes'] as $scheme) {
              if ($scheme['type'] == strtolower($interval)) {
                $limits['words'] = $scheme['words_limit'];
                $limits['images'] = $scheme['images_limit'];
                $limits['users'] = $scheme['users_limit'];
                $limits['tts'] = $scheme['tts_char_limit'];
              }
            }
          } else {
            $limits['words'] = $price['words_limit'];
            $limits['images'] = $price['images_limit'];
            $limits['users'] = $price['users_limit'];
            $limits['tts'] = $price['tts_char_limit'];
          }
        }
      }
    }

    return $limits;
  }

  /**
   * @since v1.0.0
   * Get limits of the plan
   *
   * @accepts String $pricing_id
   *
   * @return array $limits
   */
  public function getpackagePrice($pricing_id, $product_id = '')
  {
    $packs = $this->get_subscriptions();

    foreach ($packs as $k => $pack) {
      foreach ($pack['pricing'] as $key => $price) {
        if ($price['id'] == $pricing_id) {
          $type = $packs[$k]['type'];
          if ($type == 'subscription') {
            foreach ($price['schemes'] as $scheme) {
              if ($scheme['stripe_key'] == $product_id) {
                $plan_price = $scheme['price'];
              }
            }
          } else {
            $plan_price = $price['price'];
          }
        }
      }
    }

    return $plan_price;
  }

  public function paymentProcessorHandlePricing($data)
  {
    $active_methods = $this->UAI_PAYMENTS->active_payment_methods();
    $res = [];

    // PRICING FOR STRIPE
    if (in_array('stripe', $active_methods)) {
      if (isset($data['stripe_product_id']) && !empty($data['stripe_product_id'])) {
        $stripe_pricing = $this->UAI_STRIPE->createPricing($data, $this->getDefaultCurrency());

        $res['stripe_pricing'] = $stripe_pricing;
      } else {
        $res['stripe_pricing'] = [
          'status' => 401,
          'message' => esc_html__('Product exists', ULTIMATE_AI_SLUG),
        ];
      }
    }
    if (in_array('paypal', $active_methods)) {
      // PRICING FOR PAYPAL
      if (isset($data['paypal_product_id']) && !empty($data['paypal_product_id'])) {
        $paypal_pricing = $this->UAI_PAYPAL->createPricing($data, $this->getDefaultCurrency());

        $res['paypal_pricing'] = $paypal_pricing;
      } else {
        $res['paypal_pricing'] = false;
      }
    }

    return $res;
  }

  public function paymentProcessorHandlePackName($data)
  {
    $active_methods = $this->UAI_PAYMENTS->active_payment_methods();

    $res = [];

    // GET PACKAGE FOR STRIPE
    if (in_array('stripe', $active_methods)) {
      if (!isset($data['stripe_product_id']) && empty($data['stripe_product_id'])) {
        $stripe_prod = $this->UAI_STRIPE->createProduct($data['plan_name']);

        $res['stripe_product_id'] = $stripe_prod;
      } else {
        $stripe_prod = $this->UAI_STRIPE->updateProduct($data['plan_name'], $data['stripe_product_id']);

        $res['stripe_product_id'] = $stripe_prod;
      }
    }

    // GET PACKAGE FOR PAYPAL
    if (in_array('paypal', $active_methods)) {
      if (!isset($data['paypal_product_id']) && empty($data['paypal_product_id'])) {
        $paypal_prod = $this->UAI_PAYPAL->createProduct($data['plan_name']);

        $res['paypal_product_id'] = $paypal_prod;
      } else {
        $update = $this->UAI_PAYPAL->updateProduct($data['plan_name'], $data['paypal_product_id']);

        $res['paypal_product_id'] = $update;
      }
    }

    if (empty($res)) {
      return $active_methods;
    }

    return $res;
  }

  public function getDefaultCurrency()
  {
    return 'usd';
  }

  public function getCurrencySymbol($cur)
  {
    $currencySymbols = [
      'usd' => '$',
      'eur' => '€',
      'yen' => '¥',
      'gbp' => '£',
      'inr' => '₹',
      'lira' => '₤',
      'bdt' => '৳',
    ];

    return $currencySymbols[strtolower($cur)];
  }

  private function upgrade_subscription()
  {
  }

  private function downgrade_subscription()
  {
  }

  private function get_user_subscription($user_id)
  {
  }
}
